// Add this guard clause at the top of content.js 
if (window.top !== window.self) {
  // Don't run the script in an iframe
} else {
  // ALL YOUR EXISTING CODE GOES INSIDE THIS ELSE BLOCK
  // ... from "chrome.storage.local.get..." to the very end.
}

// Check for user consent before injecting the widget
chrome.storage.local.get(['dataConsent'], (result) => {
  if (result.dataConsent) {
    // If consent is true, run the original logic
    injectSenovaWidget();
  } else {
    // If consent is not given, do nothing.
    console.log('Senova AI: Data analysis consent not given. Widget not injected.');
  }
});


function injectSenovaWidget() {
    // 1. PREVENT DUPLICATE INJECTIONS
    if (window.__senovaInjected) {
      console.log('Senova widget already injected.');
      return;
    }
    window.__senovaInjected = true;
    console.log('Injecting Senova AI Widget...');
  

    let state = {
      isExpanded: true, 
      lastAnalyzedPrompt: null, 
    };
  
    async function analyzeContext(userPrompt) {
      try {
        const response = await fetch('https://senova-backend-service-644417962847.asia-south1.run.app/api/analyze', {
          method: 'POST',
          headers: { 'Content-Type': 'application/json' },
          body: JSON.stringify({ text: userPrompt })
        });
        if (!response.ok) {
          const err = await response.json();
          throw new Error(err.detail || "API request failed");
        }
        const data = await response.json();
        renderResponse(data);
      } catch (error) {
        console.error("Senova analysis failed:", error);
        renderResponse({ response: `Error: ${error.message}` });
      }
    }
  

  
    function observeChat() {
      const platformSelectors = {
        chatGPT: {
          container: 'main .h-full',
          userMessage: '[data-message-author-role="user"]'
        },
        gemini: {
          container: 'main, div[role="main"]',
          userMessage: ''
        },
        claude: {
          container: '[data-testid="chat-scroll-wrapper"]',
          userMessage: '[data-testid^="user-message-"]'
        }
      };
      

      let activePlatform = null;
      
      // --- Find the correct container for the current site ---
      let chatContainer;
      if (document.querySelector(platformSelectors.chatGPT.container)) {
          activePlatform = platformSelectors.chatGPT;
          console.log("Senova: ChatGPT platform detected.");
      } else if (document.querySelector(platformSelectors.gemini.container)) {
          activePlatform = platformSelectors.gemini;
          console.log("Senova: Gemini platform detected.");
      } else if (document.querySelector(platformSelectors.claude.container)) {
          activePlatform = platformSelectors.claude;
          console.log("Senova: Claude platform detected.");
      }

      if (!activePlatform) {
          console.warn("Senova: Could not find a compatible chat container on this page.");
          return;
      }

      chatContainer = document.querySelector(activePlatform.container);
      
      let isInitialLoad = true;
      setTimeout(() => { isInitialLoad = false; }, 2000);

      // Gemini-specific: attach input listeners to capture the user's prompt at send time
      if (activePlatform === platformSelectors.gemini && chatContainer) {
        setupGeminiInputListeners(chatContainer);
      }

      const observer = new MutationObserver((mutations) => {
        if (isInitialLoad) return; 
        
        for (const mutation of mutations) {
          if (mutation.type !== 'childList' || mutation.addedNodes.length === 0) continue;
  
          for (const node of mutation.addedNodes) {
            if (node.nodeType !== 1) continue;
  
            // Use the selector for the currently active platform
            const userMessageNode = node.querySelector(activePlatform.userMessage);
            
            if (userMessageNode) {
              const userPrompt = userMessageNode.innerText;
              
              if (userPrompt && userPrompt !== state.lastAnalyzedPrompt) {
                console.log(`Senova: Detected new user prompt: ${userPrompt}`);
                state.lastAnalyzedPrompt = userPrompt;
  
                renderLoadingView();
                analyzeContext(userPrompt);
                return; // Important: process only one prompt at a time
              }
            }
          }
        }
      });
  
      observer.observe(chatContainer, { childList: true, subtree: true });
      console.log("Senova is observing the chat...");
    }
  
  
    // Attach listeners to Gemini input to capture prompts on submit
    function setupGeminiInputListeners(root) {
      if (window.__senovaGeminiListenersAttached) return;
      window.__senovaGeminiListenersAttached = true;

      let lastDraft = '';

      function findInputElements() {
        const inputs = Array.from(root.querySelectorAll('textarea, div[contenteditable="true"]'));
        return inputs.filter(el => {
          const style = window.getComputedStyle(el);
          return style.display !== 'none' && style.visibility !== 'hidden' && el.offsetParent !== null;
        });
      }

      function getCurrentText() {
        const input = findInputElements()[0];
        if (!input) return '';
        return input.tagName.toLowerCase() === 'textarea' ? input.value : input.innerText;
      }

      function handleSubmit(source) {
        const text = lastDraft || getCurrentText();
        if (!text || text.trim() === '' || text === state.lastAnalyzedPrompt) return;
        state.lastAnalyzedPrompt = text;
        renderLoadingView();
        analyzeContext(text);
        lastDraft = '';
      }

      // Keydown on textarea/contenteditable (Enter without Shift/Ctrl)
      root.addEventListener('keydown', (e) => {
        const target = e.target;
        if (!target) return;
        const isTextArea = target.tagName && target.tagName.toLowerCase() === 'textarea';
        const isEditable = target.getAttribute && target.getAttribute('contenteditable') === 'true';
        if (!(isTextArea || isEditable)) return;

        // keep an updated draft
        lastDraft = isTextArea ? target.value : target.innerText;

        if (e.key === 'Enter' && !e.shiftKey && !e.ctrlKey && !e.metaKey && !e.altKey) {
          // Let Gemini handle sending; we just read the value
          setTimeout(() => handleSubmit('enter'), 0);
        }
      }, true);

      // Click on send button
      root.addEventListener('click', (e) => {
        const path = e.composedPath ? e.composedPath() : (e.path || []);
        const el = path.find && path.find(n => n && n.tagName && n.tagName.toLowerCase() === 'button');
        const btn = el || e.target;
        if (!(btn instanceof Element)) return;
        const isSend = btn.matches('button[aria-label*="Send" i], button[type="submit"], button[aria-label*="send" i]');
        if (isSend) {
          setTimeout(() => handleSubmit('click'), 0);
        }
      }, true);
    }

    
    function typewriterEffect(element, text, speed = 20, callback) {
        let i = 0;
        element.innerHTML = '';
        
        function type() {
          if (i < text.length) {
              element.innerHTML = text.substring(0, i + 1);
              i++;
              setTimeout(type, speed);
          } else {
              element.classList.remove('typing');
              if (callback) callback();
          }
      }
      element.classList.add('typing');
      type();
  }
    
    function renderLoadingView() {
        const contentArea = document.getElementById('senova-content-area');
        if (!contentArea) return;
        contentArea.innerHTML = `
          <div class="senova-loader">
            <div class="senova-loader-hue">
                <div class="senova-loader-hue-circle"></div>
                <div class="senova-loader-hue-circle"></div>
                <div class="senova-loader-hue-circle"></div>
                <div class="senova-loader-hue-circle"></div>
            </div>
            <p>Senova is analyzing the conversation...</p>
          </div>
        `;
      }
  
      // In content.js

  function renderResponse(data) {
    const contentArea = document.getElementById('senova-content-area');
    if (!contentArea) return;
    contentArea.innerHTML = '';

    // Create a version of the response with the topic in bold
    const boldedResponse = data.topic 
        ? data.response.replace(data.topic, `<strong>${data.topic}</strong>`)
        : data.response;

    // 1. Create and render the main insight card
    const insightCard = document.createElement('div');
    insightCard.className = 'senova-card';
    insightCard.innerHTML = `<h3>Senova's Insight</h3>`;
    const insightParagraph = document.createElement('p');
    insightParagraph.id = 'senova-insight-text';
    insightParagraph.style.marginTop = "10px";
    insightCard.appendChild(insightParagraph);
    contentArea.appendChild(insightCard);

    // 2. Start the typewriter effect
      typewriterEffect(insightParagraph, boldedResponse || "...", 20, () => {
          // This callback runs after the typewriter is finished.
          const hasQuiz = !!data.quiz;
          const hasGames = data.gameRecommendations && data.gameRecommendations.length > 0;
          
          if (!hasQuiz && !hasGames) return; // Stop if there are no actions

          // 3. Create the SECOND card for the actions/choices
          const actionCard = document.createElement('div');
          actionCard.className = 'senova-card senova-action-card fade-in';

          let actionHTML = '';

          if (hasQuiz) {
              actionHTML += `
                  <button id="senova-quiz-btn" class="senova-action-btn-stacked">
                        <span class="action-title">Start Interactive Quiz</span>
                        <span class="action-subtitle">Test your knowledge on ${data.topic || 'the topic'}.</span>
                    </button>
              `;
          }

          if (hasQuiz && hasGames) {
              actionHTML += `<div class="or-divider">OR</div>`;
          }

          if (hasGames) {
              actionHTML += `
                  <button id="senova-game-btn" class="senova-action-btn-stacked">
                        <span class="action-title">Play a Mindful Game</span>
                        <span class="action-subtitle">Enhance your cognitive abilities.</span>
                    </button>
              `;
          }

          actionCard.innerHTML = actionHTML;
          contentArea.appendChild(actionCard);

          // Add event listeners
          actionCard.querySelector('#senova-quiz-btn')?.addEventListener('click', () => showQuiz());
          actionCard.querySelector('#senova-game-btn')?.addEventListener('click', () => showGames());
          
          // --- UI-changing functions ---
          function showQuiz() {
              actionCard.remove();
              if (data.quiz) {
                  const quizCard = document.createElement('div');
                  quizCard.className = 'senova-card senova-quiz-card fade-in';
                  quizCard.appendChild(getQuizHTML(data.quiz));
                  contentArea.appendChild(quizCard);
              }
          }

          function showGames() {
            actionCard.remove();
            if (hasGames) {
                const gamesCard = document.createElement('div');
                gamesCard.className = 'senova-card senova-games-card fade-in';
                gamesCard.innerHTML = `<h3>Game Recommendations</h3>`;
                
                const gameList = document.createElement('div');
                gameList.className = 'senova-game-list';
                gameList.style.display = 'flex';
                gameList.style.flexDirection = 'column';
                gameList.style.gap = '15px';
                gameList.style.marginTop = '15px';


                data.gameRecommendations.forEach(game => {
                  const gameButton = document.createElement('button');
                  gameButton.className = 'senova-action-btn-stacked';
                  gameButton.innerHTML = `
                      <span class="action-title">${game.name}</span>
                      <span class="action-subtitle">${game.description}</span>
                  `;
              
                  gameButton.addEventListener('click', () => {
                      let gameUrl = "";
              
                      // if-else condition to set URL
                      if (game.name === "🎯 Whack-a-Thought") {
                          gameUrl = "https://dev-sameer-khan.github.io/game/games/Whack-a-thought/static/index.html";
                      } else if (game.name === "🌈 Color Maze") {
                          gameUrl = "https://dev-sameer-khan.github.io/game/games/color_maze/static/index.html";
                      } else if (game.name === "🔄 Shape Flow") {
                          gameUrl = "https://dev-sameer-khan.github.io/game/games/shape_flow/static/index.html";
                      } else if (game.name === "🎵 Breathing Maze") {
                          gameUrl = "https://dev-sameer-khan.github.io/game/games/Breathing_Maze/static/index.html";
                      } else {
                          gameUrl = "#";
                      }
              
                      contentArea.innerHTML = `
                          <div class="senova-card overflow-hidden" style="overflow: hidden ;">
                              <button id="back-to-senova" class="senova-back-btn">← Back to Senova</button>
                              <iframe style="height: 80vh" src="${gameUrl}" frameborder="0" class="senova-game-iframe" allow="fullscreen; autoplay; clipboard-write"></iframe>
                              <div class="senova-iframe-warning" style="color: #b00; font-size: 0.95em; margin-top: 10px; display: none;">
                                  If the game does not load, please <a href="${gameUrl}" target="_blank" rel="noopener noreferrer">click here to open it in a new tab</a>.
                              </div>
                              <script>
                                  const iframe = document.querySelector('.senova-game-iframe');
                                  iframe.onerror = function() {
                                      document.querySelector('.senova-iframe-warning').style.display = 'block';
                                  };
                                  // Chrome extensions block remote iframes by default, so always show warning
                                  document.querySelector('.senova-iframe-warning').style.display = 'block';
                              </script>
                          </div>
                      `;
              
                      document.getElementById('back-to-senova').addEventListener('click', () => {
                          renderResponse(data);
                      });
                  });
              
                  gameList.appendChild(gameButton);
              });
                
                gamesCard.appendChild(gameList);
                contentArea.appendChild(gamesCard);
              }
          }
      });
    }
    
    
    function getQuizHTML(quiz) {
        const quizContainer = document.createElement('div');
        quizContainer.innerHTML = `<h3>Your Custom Quiz</h3>`;

        const form = document.createElement('form');
        quiz.mcq_questions.forEach((q, index) => {
            const questionEl = document.createElement('div');
            questionEl.className = 'senova-quiz-question';
            questionEl.dataset.correctAnswer = q.answer;

            const optionsHTML = q.options.map(opt => `
                <label class="senova-quiz-option">
                    <input type="radio" name="mcq${index}" value="${opt}">
                    <span>${opt}</span>
                </label>
            `).join('');

            questionEl.innerHTML = `
                <p><strong>${index + 1}.</strong> ${q.question}</p>
                <div class="senova-options">${optionsHTML}</div>
                <div class="answer-reveal"><strong>Explanation:</strong> ${q.explanation || ''}</div>
            `;
            form.appendChild(questionEl);
        });

        const submitBtn = document.createElement('button');
        submitBtn.className = 'senova-submit-btn';
        submitBtn.type = 'submit';
        submitBtn.textContent = 'Submit Quiz';
        form.appendChild(submitBtn);

        const resultsEl = document.createElement('div');
        resultsEl.className = 'senova-quiz-results';
        form.appendChild(resultsEl);

        form.addEventListener('submit', (e) => {
            e.preventDefault();
            let score = 0;
            const questions = form.querySelectorAll('.senova-quiz-question');
            
            questions.forEach((qEl) => {
                const selected = qEl.querySelector(`input[type="radio"]:checked`);
                qEl.querySelectorAll('.senova-quiz-option').forEach(optLabel => {
                    const optInput = optLabel.querySelector('input');
                    if (optInput.value === qEl.dataset.correctAnswer) {
                        optLabel.classList.add('correct');
                    }
                    else if (selected && selected.value === optInput.value) {
                        optLabel.classList.add('incorrect');
                    }
                });

                if (selected && selected.value === qEl.dataset.correctAnswer) {
                    score++;
                }
                qEl.querySelector('.answer-reveal').style.display = 'block';
            });
            
            resultsEl.innerHTML = `<h4>Your Score: ${score} out of ${questions.length}</h4>`;
            resultsEl.style.display = 'block';
            submitBtn.style.display = 'none';
            form.classList.add('submitted');
        });

        quizContainer.appendChild(form);
        return quizContainer;
    }
  
    // =================================================================================
    // 5. WIDGET CREATION & LAYOUT ADJUSTMENT
    // =================================================================================
  
    function adjustHostLayout() {
        const mainContent = document.querySelector('main .overflow-hidden');
        if (mainContent) {
            mainContent.style.transition = 'margin-right 0.3s ease-in-out';
            mainContent.style.marginRight = '351px';
        }
    }
  
    function resetHostLayout() {
        const mainContent = document.querySelector('main .overflow-hidden');
        if (mainContent) {
            mainContent.style.marginRight = '0px';
        }
    }
  
    function createWidget() {
      const style = document.createElement('style');
      style.textContent = getWidgetCSS();
      document.head.appendChild(style);
  
      const container = document.createElement("div");
      container.id = "senova-container";
      container.innerHTML = `
        <div id="senova-header">
            <div class="senova-header-logo">
              <img src="https://res.cloudinary.com/dbgzq41x2/image/upload/v1756282806/logo_yvfsfs.svg" alt="Senova">
              <span>Senova</span>
            </div>
            <button id="senova-toggle-btn">&#9654;</button>
        </div>
        <div id="senova-content-area">
            <div class="senova-placeholder">
              <p>Chat with the AI to get started.</p>
            </div>
        </div>
         <div class="senova-footer">
          Powered by Senova AI
        </div>
      `;
      document.body.appendChild(container);
      
      document.getElementById('senova-toggle-btn').addEventListener('click', () => {
          const isCollapsed = container.classList.toggle('collapsed');
          if (isCollapsed) {
              resetHostLayout();
              document.getElementById('senova-toggle-btn').innerHTML = '&#9664;';
              document.getElementById('senova-toggle-btn').style.position = "absolute";
              document.getElementById('senova-toggle-btn').style.left = "0%";
          } else {
              adjustHostLayout();
              document.getElementById('senova-toggle-btn').innerHTML = '&#9654;';
              document.getElementById('senova-toggle-btn').style.position = "absolute"
              document.getElementById('senova-toggle-btn').style.left = "88%"
          }
      });
  
      adjustHostLayout();
    }
  
    // =================================================================================
    // 6. CSS STYLES
    // =================================================================================
  
    function getWidgetCSS() {
      return `
        :root {
          --senova-bg: rgba(26, 26, 26, 0.7);
          --senova-surface: rgba(44, 44, 44, 0.6);
          --senova-border: rgba(255, 255, 255, 0.1);
          --senova-text-primary: #F5F5F7;
          --senova-text-secondary: #A1A1A6;
          --senova-accent: #C77DFF;
          --senova-correct: rgba(46, 204, 113, 0.2);
          --senova-incorrect: rgba(231, 76, 60, 0.2);
        }
    
        #senova-container {
          position: fixed;
          top: 0;
          right: 0;
          width: 350px;
          height: 100%;
          z-index: 99999;
          display: flex;
          flex-direction: column;
          font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, sans-serif;
          color: var(--senova-text-primary);
          background-color: var(--senova-bg);
          backdrop-filter: blur(18px) saturate(180%);
          border-left: 1px solid var(--senova-border);
          background-image: radial-gradient(circle at top right, rgba(199, 125, 255, 0.12), transparent 40%);
          transition: right 0.4s cubic-bezier(0.22, 1, 0.36, 1);
        }
    
        #senova-container.collapsed {
          right: calc(-350px + 45px);
        }
    
        #senova-header {
          display: flex;
          align-items: center;
          justify-content: space-between;
          padding: 12px 0px 12px 16px;
          border-bottom: 1px solid var(--senova-border);
          flex-shrink: 0;
        }
    
        .senova-header-logo {
          display: flex;
          align-items: center;
          gap: 8px;
          font-weight: 600;
          font-size: 1.1rem;
        }
    
        .senova-header-logo img {
          width: 24px;
          height: 24px;
          filter: invert(1) brightness(2);
        }
    
        #senova-toggle-btn {
          background: transparent;
          border: none;
          color: var(--senova-text-secondary);
          cursor: pointer;
          font-size: 1.2rem;
          padding: 12px;
          height: 48px;
          border-left: 1px solid var(--senova-border);
        }
    
        #senova-toggle-btn:hover {
          color: var(--senova-text-primary);
        }
    
        #senova-container.collapsed .senova-header-logo,
        #senova-container.collapsed #senova-content-area,
        #senova-container.collapsed .senova-footer {
          opacity: 0;
          transition: opacity 0.1s;
        }
    
        #senova-content-area {
          flex-grow: 1;
          overflow-y: auto;
          padding: 16px;
          transition: opacity 0.2s;
        }
    
        .senova-footer {
          text-align: center;
          padding: 10px;
          font-size: 0.8rem;
          color: var(--senova-text-secondary);
          border-top: 1px solid var(--senova-border);
          flex-shrink: 0;
          transition: opacity 0.2s;
        }
    
        .senova-placeholder {
          text-align: center;
          color: var(--senova-text-secondary);
          padding-top: 40px;
        }
    
        .senova-loader {
          text-align: center;
          color: var(--senova-text-secondary);
          display: flex;
          flex-direction: column;
          justify-content: center;
          align-items: center;
          height: 100%;
        }
    
        .senova-loader p {
          font-size: 1.1rem;
          font-weight: 500;
          margin: 20px 0 5px 0;
        }
    
        .senova-loader-swirl {
          position: relative;
          width: 60px;
          height: 60px;
          margin-bottom: 24px;
        }
    
        .senova-loader-swirl div {
          position: absolute;
          width: 100%;
          height: 100%;
          border-radius: 50%;
          background: var(--senova-accent);
          animation: swirl 2.5s ease-in-out infinite;
        }
    
        .senova-loader-swirl div:nth-child(2) {
          animation-delay: -0.8s;
        }
    
        .senova-loader-swirl div:nth-child(3) {
          animation-delay: -1.6s;
        }

        @keyframes swirl {
          0%, 100% { transform: scale(0.2); opacity: 0.1; }
          50% { transform: scale(1); opacity: 0.6; }
        }
    
        .senova-card {
          background-color: var(--senova-surface);
          border: 1px solid var(--senova-border);
          border-radius: 12px;
          padding: 16px;
          margin-bottom: 16px;
          box-shadow: 0 4px 12px rgba(0, 0, 0, 0.2);
        }
    
        .senova-card h3 {
          margin-top: 0;
          color: var(--senova-accent);
          font-size: 1rem;
        }
    
        .senova-card p,
        .senova-card li {
          font-size: 0.95rem;
          line-height: 1.6;
          color: var(--senova-text-secondary);
          white-space: pre-wrap;
        }
    
        .senova-card ul {
          list-style-type: none;
          padding: 0;
          margin: 0;
        }
    
        .senova-actions {
          margin-top: 16px;
          padding-top: 16px;
          border-top: 1px solid var(--senova-border);
          display: flex;
          flex-direction: column;
          gap: 10px;
        }
    
        .senova-action-btn {
          width: 100%;
          padding: 12px;
          background-color: rgba(255, 255, 255, 0.05);
          border: 1px solid var(--senova-border);
          border-radius: 8px;
          color: var(--senova-text-primary);
          font-size: 0.9rem;
          font-weight: 500;
          cursor: pointer;
          text-align: left;
          transition: background-color 0.2s, border-color 0.2s;
        }
        
        .senova-action-btn-stacked {
            width: 100%;
            padding: 16px;
            background-color: rgba(255, 255, 255, 0.05);
            border: 1px solid var(--senova-border);
            border-radius: 10px;
            color: var(--senova-text-primary);
            cursor: pointer;
            text-align: left;
            transition: background-color 0.2s, border-color 0.2s;
            display: flex;
            flex-direction: column;
            gap: 4px;
        }
        .senova-action-btn-stacked:hover {
            background-color: var(--senova-surface);
            border-color: var(--senova-accent);
        }
        .action-title {
            font-size: 0.95rem;
            font-weight: 600;
            color: var(--senova-text-primary);
        }
        
        .action-subtitle {
            font-size: 0.85rem;
            color: var(--senova-text-secondary);
        }
        .or-divider {
            text-align: center;
            color: var(--senova-text-secondary);
            font-size: 0.8rem;
            font-weight: 600;
            margin: 12px 0;
        }
        .proceed-btn {
            margin-top: 16px;
            text-align: center;
            background-color: var(--senova-accent) !important;
            border-color: var(--senova-accent) !important;
        }

        .senova-loader-hue {
          position: relative;
          width: 80px;
          height: 80px;
        }
    
        .senova-loader-hue-circle {
          position: absolute;
          top: 0;
          left: 0;
          width: 100%;
          height: 100%;
          border-radius: 50%;
          background: radial-gradient(circle, transparent 10%, var(--senova-accent) 100%);
          opacity: 0;
          animation: pulse-rotate 4s ease-in-out infinite;
        }
    
        .senova-loader-hue-circle:nth-child(1) { animation-delay: 0s; }
        .senova-loader-hue-circle:nth-child(2) { animation-delay: 0.5s; transform: scale(0.8); }
        .senova-loader-hue-circle:nth-child(3) { animation-delay: 1s; transform: scale(0.6); }
        .senova-loader-hue-circle:nth-child(4) { animation-delay: 1.5s; transform: scale(0.4); }
    
        @keyframes pulse-rotate {
          0%, 100% { transform: rotate(0deg) scale(0.5); opacity: 0.1; }
          50% { transform: rotate(180deg) scale(1); opacity: 0.4; }
        }
    
        .senova-action-btn:hover {
          background-color: var(--senova-accent);
          border-color: var(--senova-accent);
        }
    
        @keyframes fadeIn {
          from { opacity: 0; transform: translateY(10px); }
          to { opacity: 1; transform: translateY(0); }
        }
    
        .senova-card.fade-in {
          animation: fadeIn 0.5s ease-in-out forwards;
        }
    
        #senova-insight-text.typing::after {
          content: '▋';
          animation: blink 0.7s infinite;
          color: var(--senova-accent);
        }
    
        @keyframes blink {
          0%, 100% { opacity: 1; }
          50% { opacity: 0; }
        }
    
        /* Styles for the interactive quiz */
        .senova-quiz-question {
          margin-top: 16px;
          padding-top: 16px;
          border-top: 1px solid var(--senova-border);
        }
    
        .senova-quiz-question:first-child {
          margin-top: 0;
          padding-top: 0;
          border-top: none;
        }
    
        .senova-options {
          display: flex;
          flex-direction: column;
          gap: 8px;
        }
    
        .senova-quiz-option {
          display: block;
          padding: 10px;
          border-radius: 8px;
          background-color: rgba(0,0,0,0.2);
          cursor: pointer;
          border: 1px solid var(--senova-border);
          transition: border-color 0.2s, background-color 0.2s;
        }
    
        .senova-quiz-option:hover {
          border-color: var(--senova-accent);
        }
    
        .senova-quiz-option input[type="radio"] {
          display: none;
        }
    
        .senova-quiz-option input[type="radio"]:checked + span {
          color: var(--senova-accent);
          font-weight: 600;
        }
    
        .senova-submit-btn {
          width: 100%;
          margin-top: 20px;
          padding: 12px;
          background-color: var(--senova-accent);
          border: none;
          border-radius: 8px;
          color: white;
          font-size: 1rem;
          font-weight: 600;
          cursor: pointer;
        }
    
        .answer-reveal {
          display: none;
          margin-top: 12px;
          padding: 10px;
          border-radius: 8px;
          background-color: rgba(0,0,0,0.2);
          font-size: 0.9rem;
          border-left: 3px solid var(--senova-accent);
        }
    
        .senova-quiz-results {
          display: none;
          margin-top: 20px;
          text-align: center;
        }
    
        .senova-quiz-option.correct {
          background-color: var(--senova-correct);
          border-color: #2ecc71 !important;
        }
    
        .senova-quiz-option.incorrect {
          background-color: var(--senova-incorrect);
          border-color: #e74c3c !important;
        }
    
        form.submitted .senova-quiz-option {
          cursor: default;
        }
    
        form.submitted .senova-quiz-option:hover {
          border-color: var(--senova-border);
        }
    
        form.submitted .senova-quiz-option.correct:hover {
          border-color: #2ecc71 !important;
        }
    
        form.submitted .senova-quiz-option.incorrect:hover {
          border-color: #e74c3c !important;
        }
      `;
    }
    
    // =================================================================================
    // 7. SCRIPT EXECUTION
    // =================================================================================
  
    if (document.readyState === 'loading') {
      document.addEventListener('DOMContentLoaded', () => {
          createWidget();
          observeChat();
      });
    } else {
      createWidget();
      observeChat();
    }
  }
  
  try {
    injectSenovaWidget();
  } catch (error) {
    console.error('Senova AI widget injection failed:', error);
  }